/*

CUSTOMIZE THE FOLLOWING TO GET YOUR OWN INDICATOR UP AND RUNNING

1) Add to the INPUT PARAMETERS the parameters you need for your signal, this can be a period for the RSI or Moving Average, the step for a PSAR and so on

2) Add some requirement checks in OnInitPreChecksPass if it is necessary to check the consistency of the parameters

3) Define in the IsSignal function at the end of the code the rule for your signal

*/

//-PROPERTIES-//
//Properties help the software look better when you load it in MT4
//Provide more information and details
//This is what you see in the About tab when you load an Indicator or an Expert Advisor
#property link          "https://www.earnforex.com/metatrader-indicators/alert-indicator-template/"
#property version       "1.0"
#property strict
#property copyright     "EarnForex.com - 2019-2021"
#property description   "Alert Indicator Template"
#property description   "So You Can Create Your Own Indicator And Alerts"
#property description   " "
#property description   "WARNING : You use this software at your own risk."
#property description   "The creator of these plugins cannot be held responsible for any damage or loss."
#property description   " "
#property description   "Find More on EarnForex.com"
//You can set an icon for the indicator to show when loading it on chart
//Icon must have an ico extension and be located in the MQL4/Files folder, the following commented line is an example of icon
#property icon          "\\Files\\EF-Icon-64x64px.ico"


//These two properties define where the indicator will be visible and how many buffers it has
//Buffers are series of values that can be read from outside the indicator for example calling
//the indicator with the iCustom MT4 function
//This template includes 4 buffers
//    1) Buy signal buffer to draw the buy signal
//    1) Sell signal buffer to draw the sell signal
//    1) Stop signal buffer to draw the neutral or stop signal
//    1) Signal buffer to easily read the signal from iCustom
#property indicator_chart_window
#property indicator_buffers 4


//-COMMENTS-//
//This is a single line comment and I do it placing // at the start of the comment, this text is ignored when compiling

/*
This is a multi line comment
it starts with /* and it finishes with the * and / like below
*/


//-ENUMERATIVE VARIABLES-//
//Enumerative variables are useful to associate numerical values to easy to remember strings
//It is similar to constants but also helps if the variable is set from the input page of the Indicator
//The text after the // is what you see in the input paramenters when the Indicator loads
//It is good practice to place all the enumberative at the start

//Defining an enumerative value for the signals
enum ENUM_TRADE_SIGNAL{
   SIGNAL_BUY=1,     //BUY
   SIGNAL_SELL=-1,   //SELL
   SIGNAL_NEUTRAL=0  //NEUTRAL
};


//Defining an enumerative shift
enum ENUM_CANDLE_TO_CHECK{
   CURRENT_CANDLE=0,    //CURRENT CANDLE
   CLOSED_CANDLE=1      //PREVIOUS CANDLE
};


//Defining an enumerative for the candle size
enum ENUM_CANDLE_SIZE{
   CANDLE_SIZE_VERYSMALL=1,   //VERY SMALL
   CANDLE_SIZE_SMALL=2,       //SMALL
   CANDLE_SIZE_MEDIUM=3,      //MEDIUM
   CANDLE_SIZE_LARGE=4,       //LARGE
   CANDLE_SIZE_VERYLARGE=5   //VERY LARGE   
};


//Defining an enumerative for the arrow distance from the candle
enum ENUM_CANDLE_DISTANCE{
   CANDLE_DISTANCE_NEAR=10,    //NEAR
   CANDLE_DISTANCE_MEDIUM=50,  //MEDIUM
   CANDLE_DISTANCE_FAR=100      //FAR
};


//-INPUT PARAMETERS-//
//The input parameters are the ones that can be set by the user when launching the Indicator
//If you place a comment following the input variable this will be shown as description of the field

input string Comment1="========================";     //MQLTA Alert Indicator Template
input string IndicatorName="MQLTA-AIT";               //Indicator Short Name

input string Comment2="========================";     //Indicator Parameters
//This is likely the only section you need to edit to adapt the indicator to your goal
//For example if you are using RSI in you indicator you can add here the input for the period and the relevant levels
//In this example we include the parameters for the two moving average
input int MASlowPeriod=25;                            //Slow Moving Average Period
input int MASlowShift=0;                              //Slow Moving Average Shift
input int MAFastPeriod=10;                            //Fast Moving Average Period
input int MAFastShift=0;                              //Fast Moving Average Shift
input ENUM_MA_METHOD MAMethod=MODE_SMA;               //Moving Average Method
input ENUM_APPLIED_PRICE MAAppliedPrice=PRICE_CLOSE;  //Moving Average Applied Price

input ENUM_CANDLE_TO_CHECK CandleToCheck=CURRENT_CANDLE;    //Candle To Use For Analysis
input int BarsToScan=500;                                   //Number Of Candles To Analyse

input string Comment_3="====================";     //Notification Options
input bool EnableNotify=false;                     //Enable Notifications Feature
input bool SendAlert=true;                         //Send Alert Notification
input bool SendApp=true;                           //Send Notification to Mobile
input bool SendEmail=true;                         //Send Notification via Email
input int WaitTimeNotify=5;                        //Wait time between notifications (Minutes)

//Arrow Style can be chosen between Wingdings and preset arrows, see following URLs for all the codes
//https://docs.mql4.com/constants/objectconstants/wingdings
//https://docs.mql4.com/constants/objectconstants/arrows
input string Comment_4="====================";     //Buffers Options
input int ArrowTypeBuy=241;                        //Code For Buy Arrow
input int ArrowTypeSell=242;                       //Code For Sell Arrow
input bool ArrowShowNeutral=false;                 //Show Stop Arrow
input int ArrowTypeStop=251;                       //Code For Stop Arrow
input color ArrowColorBuy=clrGreen;                //Color For Buy Arrow
input color ArrowColorSell=clrRed;                 //Color For Sell Arrow
input color ArrowColorStop=clrGray;                //Color For Stop Arrow
input ENUM_CANDLE_SIZE ArrowSize=CANDLE_SIZE_MEDIUM;    //Size Of The Arrows
input ENUM_CANDLE_DISTANCE CandleDistance=CANDLE_DISTANCE_NEAR;    //Arrow Distance From Candle


//-GLOBAL VARIABLES-//
//The viables included in this section are global, meaning that they can be used in any part of the code
//It is useful to add a comment to remember what is the variable for

//Here we define the 4 arrays that will be set as buffers
double BufferBuy[],BufferSell[],BufferStop[],BufferSignal[];

int Shift=0;                        //Shift is used to set if the analysis will be on the current or previous candle


//-NATIVE MT4 INDICATOR RUNNING FUNCTIONS-//

//OnInit is executed once, when the EA is loaded
//OnInit is also executed if the time frame or symbol for the chart is changed
int OnInit(void){
   IndicatorSetString(INDICATOR_SHORTNAME,IndicatorName);      //Set the indicator name
   OnInitInitialization();       //Function that you can use to initialize other variables
   if(!OnInitPreChecksPass()){   //Function to check if there are requirements that need to be met in order to run
      //If the checks are not passed the indicator will fail to load and remove itself from the chart
      return(INIT_FAILED);
   }   
   InitialiseBuffers();          //Initialize the buffers
   return(INIT_SUCCEEDED);       //Return successful initialization if all the above are completed
}


//OnCalculate runs at every tick or price change received for the current chart and has a set of default input parameters
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[]){
   
   //Initialize some internal variables that will be used for calculation
   int pos=0,upTo;

   //prev_calculated is a parameter provided by MT4 and it is important to understand it
   //prev_calculated is an integer containing the number of bars the indicator was already calculated for
   //If this is the first time the indicator is loaded on chart prev_calculated is 0
   //If the indicator is already on chart and this is only an update of the price then prev_calculated is equal to the number
   //of bars already calculated
   //prev_calculated is always 0 when the indicator is initialized
   
   //Here we say that if this is the first time the indicator runs then initialize all the buffers with empty values 
   if(prev_calculated==0){
      ArrayInitialize(BufferSignal,EMPTY_VALUE);
      ArrayInitialize(BufferBuy,EMPTY_VALUE);
      ArrayInitialize(BufferSell,EMPTY_VALUE);
      ArrayInitialize(BufferStop,EMPTY_VALUE);
   }

   //If this is the first run of the indicator then set upTo to the number of bars that we want to analyse, except one
   //Otherwise only analyse bar 0, which is the current one
   if(prev_calculated==0)
      upTo=BarsToScan-1;
   else
      upTo=0;

   //If this is the first run of the indicator then call the function DrawArrows() that as we will see later, it draws 
   //the signal arrows for all the BarsToScan
   if(prev_calculated==0){
      DrawArrows();
   }
   
   //DrawArrow(0) is the function to calculate and show the signal for a single bar, in this case bar 0, which is the current one
   //It runs at every price change in order to check if the indicator needs to be updated in the current candle
   DrawArrow(0);

   //If the notifications are enabled then check if a notification has to be sent
   if(EnableNotify)
      NotifyHit();
   
   //Return the number of bars that were calculated
   return(rates_total);
}
  

//OnDeinit runs once just before the termination of the indicator, when the indicator is unloaded from the chart
void OnDeinit(const int reason){
   CleanChart();        //Removes graphical objects from the chart
}  


//-CUSTOM INDICATOR FUNCTIONS-//
//All of the following functions we have defined to execute our custom actions
//It is a good practice to always break down big tasks in smaller sub tasks and define them as functions
//Functions are a great solution to make the code more reusable and readable


//Function to initilize some of the variables required in the indicator
void OnInitInitialization(){
   LastNotification=TimeCurrent()-WaitTimeNotify*60;     //Set the last notification time to the current time minus the "wait time" to allow notifications at the beginning
   Shift=CandleToCheck;                   //Shift defines if we want to check the signal for the closed candle (1) or the candle that is forming (0)
}


//Function for run checks of requirements for the indicator to run
bool OnInitPreChecksPass(){
   //In this section you can add some of your checks so that the indicator will not load if they are not met
   //For example, if you are creating a moving average crossover indicator you can add here a condition to
   //return false if the period of the fast moving average is greater than the period of the slow moving average

   //We check that the period for the slow moving average is greater than the period of the fast moving average
   if(MASlowPeriod<=MAFastPeriod) return false;

   return true;
}


//Function to remove graphical objects from the chart
void CleanChart(){
   //Set the Windows to 0 means used the current chart
   int Window=0;
   //Scan all the graphical objects in the current chart and delete them if their name contains the indicator name
   for(int i=ObjectsTotal(ChartID(),Window,-1)-1;i>=0;i--){
      if(StringFind(ObjectName(i),IndicatorName,0)>=0){
         ObjectDelete(ObjectName(i));
      }
   }
}


//This function is to initialize the Buffers necessary to draw the signals and store the signals
void InitialiseBuffers(){
   IndicatorBuffers(4);       //Defining the number of buffers
   IndicatorDigits(Digits());   //Defining how many digits have the values in the buffers, Digits() rerurns the number of decimals in the current price
   SetIndexStyle(0,DRAW_NONE);      //Defining to not draw anything for buffer 0
   SetIndexBuffer(0,BufferSignal);  //Associate the buffer of index zero to the array BufferSignal
   SetIndexBuffer(1,BufferBuy);     //Associate the buffer of index one to the array BufferBuy
   SetIndexShift(1,0);              //Assign a 0 shift, or delay in the drawing, to the buffer
   SetIndexLabel(1,"Buy Signal");   //Assign a label to the relative buffer
   SetIndexDrawBegin(1,0);          //Defining where to start drawing the buffer
   SetIndexStyle(1,DRAW_ARROW,ArrowTypeBuy,ArrowSize,ArrowColorBuy);    //Defining the style of the drawing for the buffer
   SetIndexArrow(1,ArrowTypeBuy);   //Defining the type of arrow to draw
   SetIndexBuffer(2,BufferSell);    //Associate the buffer of index two to the array BufferSell
   SetIndexShift(2,0);              //Assign a 0 shift, or delay in the drawing, to the buffer
   SetIndexLabel(2,"Sell Signal");  //Assign a label to the relative buffer
   SetIndexDrawBegin(2,0);          //Defining where to start drawing the buffer
   SetIndexStyle(2,DRAW_ARROW,ArrowTypeSell,ArrowSize,ArrowColorSell);  //Defining the style of the drawing for the buffer
   SetIndexArrow(2,ArrowTypeSell);  //Defining the type of arrow to draw
   SetIndexBuffer(3,BufferStop);    //Associate the buffer of index three to the array BufferStop
   SetIndexShift(3,0);              //Assign a 0 shift, or delay in the drawing, to the buffer
   SetIndexLabel(3,"Stop Signal");  //Assign a label to the relative buffer
   SetIndexDrawBegin(3,0);          //Defining where to start drawing the buffer
   //We adjust the style for BufferStop accordingly in case we do not want to draw anything for the stop/neutral signal
   if(ArrowShowNeutral==true) SetIndexStyle(3,DRAW_ARROW,ArrowTypeStop,ArrowSize,ArrowColorStop);  //Defining the style of the drawing for the buffer
   else SetIndexStyle(3,DRAW_NONE,ArrowTypeStop,ArrowSize,ArrowColorStop); //Defining the style of the drawing for the buffer
   SetIndexArrow(3,ArrowTypeStop);  //Defining the type of arrow to draw
}


//Useful ready to use function to check if the price is in a new candle, it returns true only once at the first price change received in a new candle
datetime NewCandleTime=TimeCurrent();
bool CheckIfNewCandle(){
   if(NewCandleTime==iTime(Symbol(),0,0)) return false;
   else{
      NewCandleTime=iTime(Symbol(),0,0);
      return true;
   }
}


//Notification function to send email, app or alert notifications
//Email and App parameters must be already set in the MT4 client options, MT4 > Tools > Options
//The template is configured so that you can set a number of minutes to wait between notifications, it is useful to avoid spam notifications
//The first thing is to set the LastNotification variable to the current time minus the wait time, so the first notification can happen if triggered
datetime LastNotification=TimeCurrent()-WaitTimeNotify*60;  
void NotifyHit(){
   if(!EnableNotify) return;   //If notifications are disabled no need to check for notifications, exit from the function
   if(TimeCurrent()<(LastNotification+WaitTimeNotify*60)) return; //If there was a notification recevently and "wait time" hasn't passed, don't notify
   if(!SendAlert && !SendApp && !SendEmail) return;   //If notifications aren't enabled, don't notify * a ! before a boolean variable means the opposite of it, so if SendAlert is true !SendAlert is false
   //Check if there is a signal in the current candle through the IsSignal function, 0 means for the current candle
   //IsSignal returns SIGNAL_NEUTRAL, SIGNAL_BUY, SIGNAL_SELL, depending on the signal detected
   ENUM_TRADE_SIGNAL Signal=IsSignal(0);
   //If the signal is a stop or is neutral then don't notify and exit
   if(Signal==SIGNAL_NEUTRAL) return;
   //Defining the different strings of descriptions for the different notification methods
   //It is suggested to leave the indicator name in the text so that you can understand where is the notification triggered from
   //You can customise the text at your own preference
   string EmailSubject=IndicatorName+" "+Symbol()+" Notification ";
   string EmailBody="\r\n"+AccountCompany()+" - "+AccountName()+" - "+IntegerToString(AccountNumber())+"\r\n\r\n"+IndicatorName+" Notification for "+Symbol()+"\r\n\r\n";
   string AlertText=IndicatorName+" - "+Symbol()+" Notification\r\n";
   string AppText=AccountCompany()+" - "+AccountName()+" - "+IntegerToString(AccountNumber())+" - "+IndicatorName+" - "+Symbol()+" - ";
   string Text="";
   //You can customize the message for both the buy and sell signal
   if(Signal==SIGNAL_BUY){      
      Text+="Text For Buy Signal";
   }
   if(Signal==SIGNAL_SELL){      
      Text+="Text For Sell Signal";
   }
   //Complete the text with some new line characters
   EmailBody+=Text+"\r\n\r\n";
   AlertText+=Text+"\r\n";
   AppText+=Text+"";
   //If the send alert is enabled then send the alert on screen
   if(SendAlert) Alert(AlertText);  //Using a if(SendAlert) is the short form of using if(SendAlert==true)
   //If send email is enabled then send the email
   if(SendEmail){
      if(!SendMail(EmailSubject,EmailBody)) Print("Error sending email "+IntegerToString(GetLastError()));  //If the email send fails print the error in the terminal
   }
   //If the send app is enabled then send the mobile notification
   if(SendApp){
      if(!SendNotification(AppText)) Print("Error sending notification "+IntegerToString(GetLastError()));  //If the notification fails print the error in the terminal
   }
   //Set the last notification time to the current time so that the next notification is not triggered before a wait time is finished
   LastNotification=TimeCurrent();
   //Print in the terminal that a notification was sent
   Print(IndicatorName+"-"+Symbol()+" last notification sent "+TimeToString(LastNotification));
}


//DrawArrows runs only once when the indicator is loaded of if the chart is changed to another symbol/instrument or time frame
//It scans all the bars in the chart up to the BarsToScan count and detect if there are signals
void DrawArrows(){
   //If the BarsToScan is set to zero then exit
   if(BarsToScan==0) return;
   //Set the Maximum bars to the total number of bars in the chart
   int MaxBars=Bars(Symbol(),PERIOD_CURRENT);
   //If the BarsToScan are less than the total number then set MaxBars to the number of bars to scan
   if(MaxBars>BarsToScan) MaxBars=BarsToScan;
   //Loop from the last bar to the beginning and detect the signals
   //As a reminder, index 0 is the current candle, index 1 is the previous and so on
   for(int i=MaxBars-2;i>=1;i--){
      DrawArrow(i);  //Check if at the candle of index i there is a signal
   }
}


//Function to assign the signal for candle of index i to the respective buffer
void DrawArrow(int i){
   //Assign the signal value for candle i to the variable Signal
   ENUM_TRADE_SIGNAL Signal=IsSignal(i);
   //Initially assign the value EMPTY_VALUE to all of the buffers, so that if the signal for the current candle is no longer value then it is reset
   BufferSignal[i]=EMPTY_VALUE;
   BufferStop[i]=EMPTY_VALUE;
   BufferBuy[i]=EMPTY_VALUE;
   BufferSell[i]=EMPTY_VALUE;
   
   
   //We want to position a buy or stop signal below a candle and a sell signal above the candle
   //To do this we get the low of the candle and move a few points below it in case of buy/stop signal
   //We get the high of the candle and move a few points above it in case of a sell signal
   double LowPosition=iLow(Symbol(),PERIOD_CURRENT,i)-Point()*10*CandleDistance;
   double HighPosition=iHigh(Symbol(),PERIOD_CURRENT,i)+Point()*10*CandleDistance;
   //End of position calculation
   
   //If the signal is neutral or stop at candle of index i then the BuffeSignal array at that index is set to SIGNAL_NEUTRAL, which is normally 0
   if(Signal==SIGNAL_NEUTRAL){
      BufferSignal[i]=SIGNAL_NEUTRAL;
      BufferStop[i]=LowPosition;      //Set the BufferStop at the index to value Low of the candle so the arrow is drawn below the low
   }
   //If the signal is buy at candle of index i then the BuffeSignal array at that index is set to SIGNAL_BUY, which is normally 1
   if(Signal==SIGNAL_BUY){
      BufferSignal[i]=SIGNAL_BUY;
      BufferBuy[i]=LowPosition;       //Set the BufferBuy at the index to value Low of the candle so the arrow is drawn below the low
   }
   //If the signal is sell at candle of index i then the BuffeSignal array at that index is set to SIGNAL_SELL, which is normally -1
   if(Signal==SIGNAL_SELL){
      BufferSignal[i]=SIGNAL_SELL;
      BufferSell[i]=HighPosition;     //Set the BufferSell at the index to value High of the candle so the arrow is drawn above the high
   }
}


//The IsSignal function is where you check if the candle of index i has a signal
//it can return SIGNAL_BUY=1, SIGNAL_SELL=-1,SIGNAL_NEUTRAL=0
//This functions is where you define your signal rules
ENUM_TRADE_SIGNAL IsSignal(int i){
   //Define a variable j which is the index of the candle to check, this is to consider if you are checking the current candle or the closed one
   int j=i+Shift;
   //Initialize the Signal to a neutral/stop one
   ENUM_TRADE_SIGNAL Signal=SIGNAL_NEUTRAL;   
   
   //Define all the values that you are going to need to check your signal rules first
   double MASlowCurr=iMA(Symbol(),PERIOD_CURRENT,MASlowPeriod,MASlowShift,MAMethod,MAAppliedPrice,j);
   double MASlowPrev=iMA(Symbol(),PERIOD_CURRENT,MASlowPeriod,MASlowShift,MAMethod,MAAppliedPrice,j+1);
   double MAFastCurr=iMA(Symbol(),PERIOD_CURRENT,MAFastPeriod,MAFastShift,MAMethod,MAAppliedPrice,j);
   double MAFastPrev=iMA(Symbol(),PERIOD_CURRENT,MAFastPeriod,MAFastShift,MAMethod,MAAppliedPrice,j+1);
   
   //Define the condition for your buy signal and assign SIGNAL_BUY value to the Signal variable if the condition is true
   //A BUY signal is triggered when the fast MA crosses the fast one from below
   if(MAFastCurr>MASlowCurr && MAFastPrev<MASlowPrev) Signal=SIGNAL_BUY;
   
   //Define the condition for your sell signal and assign SIGNAL_SELL value to the Signal variable if the condition is true
   //A SELL signal is triggered when the fast MA crosses the fast one from above
   if(MAFastCurr<MASlowCurr && MAFastPrev>MASlowPrev) Signal=SIGNAL_SELL;
      
   //Define the condition for your stop/neutral signal and assign SIGNAL_NEUTRAL value to the Signal variable if the condition is true
   //Defining a stop/neutral condition is not always necessary, many systems work with only Buy and Sell
   
   //Return the Signal and exit the function
   return Signal;
}

